function neostk = stk_mod(stk,process,varargin)
%A tool to manipulate the image stack.
%Syntax: [stk,stk_info] = stk_mod(stk,process,'parameters')
%Input: stk = the unmodified image stack.
%       process = the manipulation desired: crop, delete, append, replace, resize
%       parameters = the parameters used to manipulate the stack for each
%       manipulation: 
%           crop = size,location:  size = the size of crop region [x y];
%               location = the upper left vertex of the crop region [x y];
%               no paramaters activates user select mode.  You draw the region.
%           delete = frames,method: frames = a vector of frames to be deleted.
%               [x y a:b ...].
%               method = How to replace the deleted frame: white, black,
%               median, mean, none.  Default = none
%           append = stk,location: stk(:,:,x:y) = the new stack to be appended
%               location = start, end or a range.
%                   start = append before the first image of current stack
%                   end = append at the end (default)
%                   range = [20 21], append between frame 20 and 21
%           replace = stk,range: stk = the new frames to replace the old
%               start = 11 the start of the replaced range or original [x y
%               a:b] & new [X Y A:B].  Remember the two frame sets must have
%               the same number of frames.
%           resize = size,method,: size = in percentage the new image, e.g., 0.8(80%), or
%               'fit', which fits the image in to the set screen size, or x
%               y sizes could be given, this can distort the image, or
%               'manual', let the user choose interactively
%               method = the method by which the image is reduced in size.
%               Parameters: 'bicubic' (default), 'bilinear','nearest' (nearest
%               neighbor)
%           extract = frames: frames = a vector of frames to be extracted
%               to form its own stack. [x y a:b...] or EvO : 'odds','evens'
%               [1 3]. odds = all odd frames.  evens = all even frames.
%               [1 3] = start on frame 1, every 3rd frame.
%               syntax: x = stk_mod(stk,'extract',[1 2 3 4 5:10] or 'odds' or
%               'EvO',[1 3]);
%Output:  stk = the modified image stack.

[x,y,z,parameter] = parse(process,stk,varargin);


switch process
    case 'crop'
        size_x = parameter.size(1,1);
        size_y = parameter.size(1,2);
        loc_x = parameter.location(1,1);
        loc_y = parameter.location(1,2);
        %Initiate the stack matrixes
        neostk = zeros(size_x,size_y,z,'uint8');        %Initiated uint8 matrix, for imshow compatability

        %Process the images into our stack format
        neostk = stk(loc_x:loc_x+size_x-1,loc_y:loc_y+size_y-1,:);  %Write cropped stack matrix
    case 'delete'
        %Initiate the variables
        del = sort(parameter.frames);         %Sort the vector of deleted frames
        del_idx = [1:z];                      %Initiate the deletion index
        del_idx(del) = [];                    %Set the deletion index
        del_num = size(del,2);                %get the # of deleted frames

        %make the replacement frame if needed
        switch parameter.method
            case 'white'
                rep_frame = ones(x,y,del_num,'uint8').*255; %White is 255 for a uint8 image, for other depths, it will change
            case 'black'
                rep_frame = zeros(x,y,del_num,'uint8');
            case 'median'       
                rep_frame = ones(x,y,del_num,'uint8').*(median(median(median(stk))));
            case 'mean'
                rep_frame = ones(x,y,del_num,'uint8').*(mean(mean(mean(stk))));
        end
        
        %delete/replace frames
        switch parameter.method
            case 'none'
                neostk = zeros(x,y,z-del_num,'uint8');        %Initiated matrix
                neostk = stk(:,:,del_idx);
                %neostk = stk(:,:,del_idx);
            otherwise
                neostk = zeros(x,y,z,'uint8');
                neostk = stk;
                neostk(:,:,del) = rep_frame;
            end
    case 'append'
        %Initialize and Setup
        insert_num = size(parameter.stk,3);
        neostk = zeros(x,y,z+insert_num,'uint8');        %Initiated matrix
        
        %append
        switch num2str(parameter.location)
            case 'end'
                neostk(:,:,1:z) = stk;
                neostk(:,:,z+1:z+insert_num) = parameter.stk;
            case 'start'
                neostk(:,:,1:insert_num) = parameter.stk;
                neostk(:,:,insert_num+1:z+insert_num) = stk;
            otherwise
                neostk(:,:,1:parameter.location(1,1)) = stk(:,:,1:parameter.location(1,1));
                neostk(:,:,parameter.location(1,1)+1:parameter.location(1,1)+insert_num) = parameter.stk;
                neostk(:,:,parameter.location(1,1)+insert_num+1:z+insert_num) = stk(:,:,parameter.location(1,2):z);
        end
    case 'replace'
        %Initialize and Setup
        insert_num = size(parameter.stk,3);
        neostk = zeros(x,y,z,'uint8');        %Initiated matrix
        
        %replace
        neostk = stk;
        neostk(:,:,parameter.original) = parameter.stk(:,:,parameter.new);
    case 'resize'
        %Initialize and setup
        [x2,y2] = size(imresize(stk(:,:,1),parameter.size,parameter.method)); %new size
        neostk = zeros(x2,y2,z,'uint8');
        
        %resize
        for i = 1:z
            neostk(:,:,i) = imresize(stk(:,:,i),parameter.size,parameter.method);
            disp(['*',num2str(round((i/z)*100)),'%']);
        end
        disp(['Done!'])
    case 'extract'
        %Initialize and Setup
        neostk = zeros(x,y,size(parameter.frames,2),'uint8');
        
        %extract it
        neostk = stk(:,:,parameter.frames);
end

%--------------------------------------------------------------------------
%subfunction to parse the inputs.
function [x,y,z,parameter] = parse(process,stk,input)

[x,y,z] = size(stk);

switch process
    case 'crop'
        %Parse the input
        if ~isempty(input)
            parameter.size = input{1,1};
            parameter.location = input{1,2};
            %error check
            if size(parameter.size,2)~=2||size(parameter.location,2)~=2
                error(['Error: You must enter two values in form [x y] for the size and location designations']);
            end
            if x+1<parameter.size(1,1)+parameter.location(1,1)||y+1<parameter.size(1,2)+parameter.location(1,2)     %Check if the crop size is beyond the image size
                error(['Error: Your specified crop size of ',num2str(parameter.size(1,1)),'x',num2str(parameter.size(1,2)),...
                    ' is beyond the image size of the stack: ',...
                    num2str(x),'x',num2str(y),'.'])       
            end

        else
            figure('units','pixels','position',[20 35 y x],'name','Crop the Picture','menubar','none','toolbar','none');
            figure1 = gcf;      %Get figure1's handle
            img = flatview(stk,'median');
            image(img);
            colormap(gray(256));
            axis off;
            set(gca,'position',[0 0 1 1]);
            [roi_mask,loc_x,loc_y] = roirect;
            delete(figure1);        %close figure1
            parameter.location = [loc_y(1,1) loc_x(1,1)];  %the location of the upper left vertex
            parameter.size = [(max(loc_y)-min(loc_y)) (max(loc_x)-min(loc_x))];   %The size of the cropped region
        end
    case 'delete'
        parameter.frames = input{1,1};
        if size(input,2)==1
            parameter.method = 'none';
        else
            parameter.method = input{1,2};
        end
        %error check
        if max(parameter.frames)>z||~isnumeric(parameter.frames)
            error(['Error: A value you entered for a frame is not within the index of the stack, or you entered a none valide frames format']);
        end
    case 'append'
        parameter.stk = input{1,1};
        if size(input,2)==1
            parameter.location = 'end';
        else
            parameter.location = input{1,2};
        end
        %error check
        if size(parameter.stk,1)~=x||size(parameter.stk,2)~=y
            error(['Error: The images in your stack to be appended must be the same size as the images in the original stack']);
        end
        if ~ischar(parameter.location)
            parameter.location = sort(parameter.location);  %Sort it just in case someone is really stupid.
            if size(parameter.location,2)~=2
                error(['Error: The location of the insertion must either be ','start',' end ','or a range [a b]']);
            elseif parameter.location(1,1)>z||parameter.location(1,2)>z
                error(['Error: Your insertion range must exist within the stack']);
            end
        end
    case 'replace'
        parameter.stk = input{1,1};
        frames = size(parameter.stk,3);
        if size(input,2)==3
            parameter.original = input{1,2};
            parameter.new = input{1,3};
        else
            parameter.original = [input{1,2}:input{1,2}+frames-1];
            parameter.new = [1:frames];
        end
        %error check
        if size(parameter.stk,1)~=x||size(parameter.stk,2)~=y
            error(['Error: The images in your stack to be appended must be the same size as the images in the original stack']);
        end
        parameter.original = sort(parameter.original);  %Sort it just in case someone is really stupid.
        parameter.new = sort(parameter.new);
        if size(parameter.original,2)~=size(parameter.new,2)
            error(['Error: Your # of replacement frames must equal the replaced frames']);
        elseif max(parameter.original)>z||max(parameter.new)>z
            error(['Error: Your insertion range must exist within the stack.']);
        elseif max(parameter.new)>frames
            error(['Error: You have exceeded the range of your input stack.']);
        end
    case 'resize'
        parameter.size = input{1,1}
        if size(input,2)==1
            parameter.method = 'bicubic';
        else
            parameter.method = input{1,2};
        end
        %size definition & error check
        if ~ischar(parameter.size)
            if size(parameter.size,2)>2
                error(['Error: for the resizeing you must either enter a percentage, e.g., 0.8, or row x column, e.g., [200 300], or string, e.g., fit']);
            end
        else
            switch parameter.size
                case 'fit'      %Fit to screen
                    screen_sz = get(0,'screensize');    %remember this returns the screen size in pixel coordinates
                    %Note: the size returned is [1 1 x y], but in matrix
                    %terms it is [1 1 col row]
                    default(1,1) = floor((screen_sz(1,3)/x)*100)/100;  %Round it a bit
                    default(1,2) = floor((screen_sz(1,4)/y)*100)/100;
                    parameter.size = min(default); %Make sure the image fits the screen in both x and y.
                case 'manual'
                    img = stk(:,:,1);
                    %default value
                    screen_sz = get(0,'screensize');
                    default(1,1) = floor((screen_sz(1,3)/x)*100)/100;  %Round it a bit
                    default(1,2) = floor((screen_sz(1,4)/y)*100)/100;
                    default = min(default); %Make sure the image fits the screen in both x and y.
                    while 1
                        scale = str2num(response_box('title','Resize the picture','input',num2str(default),'caption','the resize %, e.g., 0.8(80%)','position','center'));
                        img = imresize(img,scale);
                        [x2 y2] = size(img);
                        figure('units','pixels','position',[20 35 y2 x2],'name','Resize Image','menubar','none','toolbar','none');
                        figure1 = gcf;      %Get figure1's handle
                        image(img);
                        colormap(gray(256));
                        axis off;
                        set(gca,'position',[0 0 1 1]);
                        done = yes_no_box('title','Are you done?','caption1','Is this the picture size you want',...
                             'caption2','press yes if you are.','position','center');
                        delete(figure1)
                        if done==1
                            break
                        end
                    end
                    parameter.size = scale;
            end
        end
        %error check
        if ~ischar(parameter.method)
            error(['Error: You must enter a string for the method of resizing, e.g., bicubic.']);
        end
    case 'extract'
        parameter.frames = input{1,1};
        %error check and frames settings
        if ~ischar(parameter.frames)
            parameter.frames = sort(parameter.frames);  %make sure the frames are in order
        else
            switch parameter.frames
                case 'odds'
                    parameter.frames = [1:2:z];
                case 'evens'
                    parameter.frames = [2:2:z];
                case 'EvO'
                    index = input{1,2};
                    if size(index,2)==2
                        parameter.frames = [index(1,1):index(1,2):z];
                    else
                        error(['Error: Help is what you need.']);
                    end
                otherwise
                    error(['Error: Maybe you should check the help.']);
            end
        end
    otherwise
        error(['Error: You did not enter a valid process.  Please check help.']);
end